/*****************************************************************************************
    pngcat ver.1.2.0: PNG utility specified CGI image-counter
        Copyright 1997-2000, JoJo's Web Lab. <webmaster@starwars.jp>
        http://www.starwars.jp/web/
    "pngcat" uses the following libraries.
    gd: Thomas Boutell <boutell@boutell.com>
        Copyright 1994-1998, Quest Protein Database Centre, Cold Spring Harbour Labs
    libpng:
        Copyright 1995-1996, Guy Eric Schalnat, Group 42, Inc.
        Copyright 1996-1997, Andreas Dilger
        Copyright 1998-1999, Glenn Randers-Pehrson
    zlib:
        Copyright 1995-1998, Jean-loup Gailly.
    "pngcat" is Freesoft.


    use: pngcat [Number] [-i PngImageDirectry] [-o OutputFile] [-h]

        Number: counter number, default is 0.
        -i: default is current.
        -o: default is stdout.
        -h: view help mesage.

*****************************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <fcntl.h>
#ifdef _FOR_WINDOWS_        /* For Windows */
#include <io.h>
#endif
#include "gd.h"

#define OPTION_NONE     (0)
#define OPTION_INPUT    (1)
#define OPTION_OUTPUT   (2)
#define FILE_NOT_FOUND  (-1)
#define FILE_OPEN_FAIL  (-2)
#define SIZE_DIFFERENT  (-3)
#define INTERNAL_ERROR  (-4)
#define UNKNOWN_OPTION  (-99)


int usage(void);
int pngcat(char *number, char *dir, char *filename);

void main(int argc, char *argv[])
{
    int i;
    int ret = 0;
    int len;
    int option = OPTION_NONE;
    char *number = NULL;
    char *dir = NULL;
    char *filename = NULL;
    char *erroption = NULL;

    for (i = 1; i < argc; i++) {
        if (option != OPTION_NONE) {
            switch (option) {
                case OPTION_INPUT:
                    dir = argv[i];
                    break;
                case OPTION_OUTPUT:
                    filename = argv[i];
                    break;
                default:
                    break;
            }
            option = OPTION_NONE;
            continue;
        }
        if (argv[i][0] == '-') {
            switch (argv[i][1]) {
                case 'i':
                    if (argv[i][2] != '\0') {
                        dir = argv[i] + 2;
                    } else {
                        option = OPTION_INPUT;
                    }
                    break;
                case 'o':
                    if (argv[i][2] != '\0') {
                        filename = argv[i] + 2;
                    } else {
                        option = OPTION_OUTPUT;
                    }
                    break;
                case 'h':
                    if (argv[i][2] == '\0') {
                        ret = usage();
                        exit(0);
                    } else {
                        erroption = argv[i];
                        ret = UNKNOWN_OPTION;
                    }
                    break;
                default:
                    erroption = argv[i];
                    ret = UNKNOWN_OPTION;
                    break;
            }
        } else if (argv[i][0] >= '0' && argv[i][0] <= '9') {
            number = argv[i];
        }
    }

    if (ret < 0) {
        goto ERROR;
    }

    if (number != NULL) {
        len = strlen(number);
        for (i = 0; i < len; i++) {
            if (number[i] < '0' || number[i] > '9') {
                number[i] = '\0';
                break;
            }
        }
    }
    if (dir != NULL) {
        len = strlen(dir);
        if (len) {
            if (dir[len-1] == '/') {
                dir[len-1] = '\0';
            }
        }
    }
    if (filename != NULL) {
        len = strlen(filename);
        if (len) {
            if (filename[len-1] == '/') {
                filename[len-1] = '\0';
            }
        }
    }

    ret = pngcat(number, dir, filename);
    if (ret < 0) {
        goto ERROR;
    }

#ifdef _DEBUG_
    printf("number   = [%s]\n", (number == NULL ? "(NULL)" : number));
    printf("dir      = [%s]\n", (dir == NULL ? "(NULL)" : dir));
    printf("filename = [%s]\n", (filename == NULL ? "(NULL)" : filename));
#endif

    exit(0);

ERROR:

    switch (ret) {
        case FILE_NOT_FOUND:
            if (dir == NULL) {
                dir = ".";
            }
            fprintf(stderr, "pngcat: \"%s/*.png\" not found.\n", dir);
            break;
        case FILE_OPEN_FAIL:
            if (filename == NULL) {
                fprintf(stderr, "pngcat: can't output png image file.\n");
            } else {
                fprintf(stderr, "pngcat: \"%s\" can't create.\n", filename);
            }
            break;
        case UNKNOWN_OPTION:
            if (erroption == NULL) {
                erroption = "";
            }
            fprintf(stderr, "pngcat: %s is unknown option.\n", erroption);
            break;
        case SIZE_DIFFERENT:
            if (erroption == NULL) {
                erroption = "";
            }
            fprintf(stderr, "pngcat: image size is different.\n");
            break;
        case INTERNAL_ERROR:
            if (erroption == NULL) {
                erroption = "";
            }
            fprintf(stderr, "pngcat: memory error.\n");
            break;
        default:
            fprintf(stderr, "error.\n");
            break;
    }
    exit(-1);
}

int pngcat(char *number, char *dir, char *filename)
{
    int i;
    int ret = 0;
    int len;
    int trans_color = -1;
    int pos = 0;
    int width = 0;
    int height = 0;
    char pngfile[BUFSIZ];
    FILE *src = NULL;
    FILE *image = NULL;
    gdImagePtr *in = NULL;
    gdImagePtr out = NULL;

    if (number == NULL) {
        number = "0";
    }
    if (dir == NULL) {
        dir = ".";
    }
    len = strlen(number);
    in = (gdImagePtr*)malloc(sizeof(gdImagePtr) * len);
    if (in == NULL) {
        ret = INTERNAL_ERROR;
        goto ERROR_LINE;
    }
    for (i = 0; i < len; i++) {
        in[i] = NULL;
    }
    for (i = 0; i < len; i++) {
        sprintf(pngfile, "%s/%c.png", dir, number[i]);
        src = fopen(pngfile, "rb");
        if (src == NULL) {
            ret = FILE_NOT_FOUND;
            goto ERROR_LINE;
        }
        in[i] = gdImageCreateFromPng(src);
        fclose(src);
        src = NULL;
        width += in[i]->sx;
    }
    height = in[0]->sy;
    trans_color = in[0]->transparent;
    out = gdImageCreate(width, height);
    for (i = 0; i < len; i++) {
        if (in[i]->sy != height) {
            ret = SIZE_DIFFERENT;
            goto ERROR_LINE;
        }
        in[i]->transparent = -1;
        gdImageCopy(out, in[i], pos, 0, 0, 0, in[i]->sx, height);
        pos += in[i]->sx;
        gdImageDestroy(in[i]);
        in[i] = NULL;
    }
    gdImageColorTransparent(out, trans_color);
    if (filename != NULL) {
        image = fopen(filename, "wb");
        if (image == NULL) {
            ret = FILE_OPEN_FAIL;
            goto ERROR_LINE;
        }
        gdImagePng(out, image);
        fclose(image);
        image = NULL;
    } else {

#ifdef _FOR_WINDOWS_        /* For Windows */
        setmode(fileno(stdout), O_BINARY);
#endif

        gdImagePng(out, stdout);
    }

ERROR_LINE:

    for (i = 0; i < len; i++) {
        if (in[i] != NULL) {
            gdImageDestroy(in[i]);
        }
    }
    if (in != NULL) {
        free(in);
    }
    if (out != NULL) {
        gdImageDestroy(out);
    }
    if (src != NULL) {
        fclose(src);
    }
    if (image != NULL) {
        fclose(image);
    }
    return ret;
}

int usage(void)
{
    printf("\n");
    printf("pngcat ver.1.2.0 Copyright(c) 1997-2001, JoJo's Web Lab.\n");
    printf("\n");
    printf("pngcat [Number] [-i PngImageDirectry] [-o OutputFile] [-h]\n");
    printf("Number: counter number, default is 0.\n");
    printf("    -i: default is current.\n");
    printf("    -o: default is stdout.\n");
    printf("    -h: view help mesage.\n");
    printf("\n");

    return (0);
}
/* End Of File */
